document.addEventListener("DOMContentLoaded", () => {
    document.querySelectorAll(".warning-notice .button-primary").forEach((button) => {
        button.addEventListener("click", function () {
            let warningElement = this.closest(".warning-notice");
            let warningKey = warningElement.getAttribute("data-warning");

            fetch(ajaxurl, {
                method: "POST",
                headers: { "Content-Type": "application/x-www-form-urlencoded" },
                body: new URLSearchParams({
                    action: "NEXSOL_WS_dismiss_warning",
                    warning: warningKey,
                    NEXSOL_WS_warning_nonce: wsDismissWarning.nonce,
                }),
            })
                .then((response) => response.json())
                .then((data) => {
                    if (data.success) {
                        warningElement.style.display = "none";
                    }
                })
                .catch((error) => console.error("Error dismissing warning:", error));
        });
    });

    const form = document.querySelector(".wordsentinel-settings-form");
    const saveButton = form?.querySelector(".button-primary");

    if (!form || !saveButton) {
        return;
    }

    saveButton.disabled = true;

    form.addEventListener("input", () => {
        saveButton.disabled = false;
    });

    form.addEventListener("change", () => {
        saveButton.disabled = false;
    });

    form.addEventListener("submit", (event) => {
        const formData = new FormData(form);
        formData.append("action", "NEXSOL_WS_reset_warning");
        formData.append("nonce", wsResetWarning.nonce);

        fetch(ajaxurl, {
            method: "POST",
            body: formData,
        })
            .then((response) => response.json())
            .catch((error) => console.error("AJAX Error:", error));
    });

    const apiUrl = NEXSOL_WS_Api.gradesEndpoint;
    const isLocalhost = NEXSOL_WS_Api.isLocalhost;
    const mozillaPanel = document.getElementById("panel-mozilla");
    const sslLabsPanel = document.getElementById("panel-ssl-labs");

    if (isLocalhost) {
        if (mozillaPanel) {
            mozillaPanel.className = "wordsentinel-panel wordsentinel-panel-disabled";
            mozillaPanel.innerHTML = `
                <h3>Mozilla Observatory</h3>
                <p class="wordsentinel-message">${NEXSOL_WS_Translations.unavailable}</p>
            `;
        }

        if (sslLabsPanel) {
            sslLabsPanel.className = "wordsentinel-panel wordsentinel-panel-disabled";
            sslLabsPanel.innerHTML = `
                <h3>Qualys SSL Labs</h3>
                <p class="wordsentinel-message">${NEXSOL_WS_Translations.unavailable}</p>
            `;
        }

        return;
    }

    fetch(apiUrl, {
        method: "GET",
        headers: {
            "Content-Type": "application/json",
        },
    })
        .then((response) => {
            if (!response.ok) {
                throw new Error(`HTTP error! Status: ${response.status}`);
            }
            return response.json();
        })
        .then((data) => {
            updatePanels(data);
        })
        .catch((error) => {
            console.error("Fetch error:", error);
            document.querySelectorAll(".wordsentinel-loading").forEach((panel) => {
                panel.innerHTML =
                    '<p class="wordsentinel-message error">Error loading data. Please try again later.</p>';
            });
        });
});

function setPanelContent(panel, result, serviceName, translations, retryCallback) {
    let countdown = 60;

    function startCountdown(reloadIconContainer, timerElement) {
        reloadIconContainer.classList.add('disabled');
        const reloadIcon = reloadIconContainer.querySelector('.wordsentinel-reload-icon');
        reloadIcon.classList.add('disabled');
        const interval = setInterval(() => {
            if (countdown > 0) {
                countdown--;
                timerElement.textContent = `(${countdown}s)`;
            } else {
                clearInterval(interval);
                reloadIconContainer.classList.remove('disabled');
                reloadIcon.classList.remove('disabled');
                timerElement.textContent = '';
            }
        }, 1000);
    }

    const viewDetailsLink =
        serviceName === 'Mozilla Observatory'
            ? `https://observatory.mozilla.org/analyze/${result.site}`
            : serviceName === 'Qualys SSL Labs'
                ? `https://www.ssllabs.com/ssltest/analyze.html?d=${result.site}`
                : '#';

    if (result.status === 'success') {
        const gradeClass = getGradeClass(result.grade);
        panel.className = `wordsentinel-panel wordsentinel-panel-${gradeClass}`;
        panel.innerHTML = `
            <h3>${serviceName}</h3>
            <div class="wordsentinel-grade-circle">
                <span>${result.grade || 'N/A'}</span>
            </div>
            <div class="wordsentinel-actions">
                <a href="${viewDetailsLink}" target="_blank" class="wordsentinel-view-details-link">
                    ${translations.viewDetails}
                </a>
                <div class="wordsentinel-scan-section">
                    <span class="wordsentinel-timer"></span>
                    <div class="wordsentinel-reload-icon-container">
                        <span class="wordsentinel-reload-icon">&#x21bb;</span>
                    </div>
                </div>
            </div>`;
    } else {
        panel.className = 'wordsentinel-panel wordsentinel-panel-error';
        panel.innerHTML = `
            <h3>${serviceName}</h3>
            <p class="wordsentinel-message">${result.message || translations.errorOccurred}</p>
            <div class="wordsentinel-actions">
                <div class="wordsentinel-scan-section">
                    <span class="wordsentinel-timer"></span>
                    <div class="wordsentinel-reload-icon-container">
                        <span class="wordsentinel-reload-icon">&#x21bb;</span>
                    </div>
                </div>
            </div>`;
    }

    const reloadIconContainer = panel.querySelector('.wordsentinel-reload-icon-container');
    const timerElement = panel.querySelector('.wordsentinel-timer');

    startCountdown(reloadIconContainer, timerElement);

    reloadIconContainer.addEventListener('click', () => {
        if (!reloadIconContainer.classList.contains('disabled')) {
            retryCallback();
        }
    });
}

function setLoadingState(panel, serviceName) {
    panel.className = 'wordsentinel-panel';
    panel.innerHTML = `
        <h3>${serviceName}</h3>
        <div class="wordsentinel-loading">
            <div class="wordsentinel-loading-spinner"></div>
        </div>
    `;
}

function updatePanels(data) {
    const mozillaPanel = document.getElementById('panel-mozilla');
    const sslLabsPanel = document.getElementById('panel-ssl-labs');

    setLoadingState(mozillaPanel, 'Mozilla Observatory');
    setLoadingState(sslLabsPanel, 'Qualys SSL Labs');

    function retryMozilla() {
        setLoadingState(mozillaPanel, 'Mozilla Observatory');
        fetch(NEXSOL_WS_Api.gradesEndpoint)
            .then(response => response.json())
            .then(newData => {
                if (newData['Mozilla Observatory']) {
                    setPanelContent(
                        mozillaPanel,
                        newData['Mozilla Observatory'],
                        'Mozilla Observatory',
                        NEXSOL_WS_Translations,
                        retryMozilla
                    );
                }
            })
            .catch(err => console.error('Retry failed:', err));
    }

    function retrySslLabs() {
        setLoadingState(sslLabsPanel, 'Qualys SSL Labs');
        fetch(NEXSOL_WS_Api.gradesEndpoint)
            .then(response => response.json())
            .then(newData => {
                if (newData['Qualys SSL Labs']) {
                    setPanelContent(
                        sslLabsPanel,
                        newData['Qualys SSL Labs'],
                        'Qualys SSL Labs',
                        NEXSOL_WS_Translations,
                        retrySslLabs
                    );
                }
            })
            .catch(err => console.error('Retry failed:', err));
    }

    if (data['Mozilla Observatory']) {
        setPanelContent(
            mozillaPanel,
            data['Mozilla Observatory'],
            'Mozilla Observatory',
            NEXSOL_WS_Translations,
            retryMozilla
        );
    }

    if (data['Qualys SSL Labs']) {
        setPanelContent(
            sslLabsPanel,
            data['Qualys SSL Labs'],
            'Qualys SSL Labs',
            NEXSOL_WS_Translations,
            retrySslLabs
        );
    }
}

function getGradeClass(grade) {
    if (!grade) return 'error';
    let newGradeClass = grade.toLowerCase().replace('-', '-minus').replace('+', '-plus');
    return newGradeClass;
}
