<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (defined('WP_INSTALLING') && WP_INSTALLING) {
    return;
}

class NEXSOL_WS_API_Helper {
    /**
     * Fetches the security grade from Mozilla Observatory.
     *
     * @param string $url The URL to analyze.
     * @return array An associative array with the grade and status.
     */
    public function fetch_mozilla_observatory_grade($url) {
        $api_url = 'https://observatory-api.mdn.mozilla.net/api/v2/scan?host=' . urlencode($url);

        // Initiate the scan with a POST request
        $response = wp_remote_post($api_url, [
            'method'    => 'POST',
            'timeout'   => 20,
            'headers'   => [
                'Accept' => 'application/json',
            ],
        ]);

        if (is_wp_error($response)) {
            return ['status' => 'error', 'message' => 'Could not initiate scan on Mozilla Observatory'];
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        // Check for errors in the response
        if (isset($data['error'])) {
            return ['status' => 'error', 'message' => 'Error from Mozilla Observatory: ' . $data['error']];
        }

        // Ensure the scan has completed
        if (isset($data['grade'])) {
            return ['status' => 'success', 'grade' => $data['grade']];
        } else {
            return ['status' => 'error', 'message' => 'Scan in progress or grade not available. Please try again later.'];
        }
    }       

    /**
     * Fetches the security grade from Qualys SSL Labs.
     *
     * @param string $url The URL to analyze.
     * @return array An associative array with the grade and status.
     */
    public function fetch_ssl_labs_grade($url) {
        $api_url = 'https://api.ssllabs.com/api/v3/analyze?host=' . urlencode($url) . '&fromCache=on';
        $response = wp_remote_get($api_url);

        if (is_wp_error($response)) {
            return ['status' => 'error', 'message' => 'Could not fetch grade from Qualys SSL Labs'];
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!empty($data['status']) && $data['status'] === 'READY') {
            if (!empty($data['endpoints'][0]['grade'])) {
                return ['status' => 'success', 'grade' => $data['endpoints'][0]['grade']];
            } else {
                return ['status' => 'error', 'message' => 'Grade not found in response from Qualys SSL Labs'];
            }
        } elseif (!empty($data['status']) && $data['status'] === 'IN_PROGRESS') {
            return ['status' => 'error', 'message' => 'SSL Labs analysis in progress. Please try again later.'];
        }

        return ['status' => 'error', 'message' => 'Unexpected response from Qualys SSL Labs'];
    }
}
