<?php
/**
 * @package wordsentinel
 */
/*
Plugin Name: WordSentinel
Plugin URI: https://nexsol.tech/our-products/wordsentinel/
Description: Secure WordPress by configuring essential security headers and analyzing the website's grades via external services.
Version: 1.0
Requires at least: 5.8
Requires PHP: 7.0
Author: Nexsol Technologies Team
Author URI: https://nexsol.tech/
License: GPLv3
License URI: https://www.gnu.org/licenses/gpl-3.0.html
Text Domain: wordsentinel
Domain Path: /languages

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <https://www.gnu.org/licenses/>.

*/
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if(defined('WP_INSTALLING') && WP_INSTALLING){
	return;
}

define('NEXSOL_WS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('NEXSOL_WS_PLUGIN_URL', plugin_dir_url(__FILE__));

// Enqueue styles and scripts
add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook === 'toplevel_page_NEXSOL_WS_wordsentinel') {
        wp_enqueue_style(
            'wordsentinel-admin-style',
            plugin_dir_url(__FILE__) . 'assets/admin-style.css',
            [],
            filemtime(plugin_dir_path(__FILE__) . 'assets/admin-style.css')
        );

        wp_enqueue_script(
            'wordsentinel-admin-script',
            plugin_dir_url(__FILE__) . 'assets/admin-script.js',
            [],
            '1.0',
            true
        );

        // Detect if site is running on localhost
        $siteUrl = wp_parse_url(get_site_url(), PHP_URL_HOST);
        $isLocalhost = (strpos($siteUrl, 'localhost') !== false);

        // Pass necessary data to JS
        wp_localize_script('wordsentinel-admin-script', 'NEXSOL_WS_Api', [
            'gradesEndpoint' => rest_url('NEXSOL_WS/v1/grades'),
            'isLocalhost' => $isLocalhost,
        ]);

        wp_localize_script('wordsentinel-admin-script', 'NEXSOL_WS_Translations', [
            'viewDetails' => __('View Details', 'wordsentinel'),
            'errorOccurred' => __('An unknown error occurred.', 'wordsentinel'),
            'launchScan' => __('Launch Scan', 'wordsentinel'),
            'unavailable' => __('Unavailable in development mode.', 'wordsentinel'),
        ]);

        wp_localize_script('wordsentinel-admin-script', 'wsResetWarning', [
            'nonce' => wp_create_nonce('NEXSOL_WS_reset_warning_nonce'),
        ]);

        wp_localize_script('wordsentinel-admin-script', 'wsDismissWarning', [
            'nonce' => wp_create_nonce('NEXSOL_WS_warning_action'),
        ]);
    }
});

// REST API Endpoint for Grades
add_action('rest_api_init', function () {
    register_rest_route('NEXSOL_WS/v1', '/grades', [
        'methods' => 'GET',
        'callback' => 'NEXSOL_WS_fetch_grades',
        'permission_callback' => function () {
            // Allow only logged-in users with specific roles
            $allowed_roles = ['administrator', 'editor'];
            $user = wp_get_current_user();
            return array_intersect($allowed_roles, $user->roles);
        },
    ]);
});

add_action('wp_ajax_NEXSOL_WS_dismiss_warning', 'NEXSOL_WS_dismiss_warning');
add_action('wp_ajax_NEXSOL_WS_reset_warning', 'NEXSOL_WS_reset_warning');

function NEXSOL_WS_dismiss_warning() {
    $user_id = get_current_user_id();
    if (!$user_id) {
        wp_send_json_error(['message' => __('User not logged in', 'wordsentinel')]);
        exit;
    }

    if (isset($_POST['NEXSOL_WS_warning_nonce']) && check_admin_referer('NEXSOL_WS_warning_action', 'NEXSOL_WS_warning_nonce')) {
        if (isset($_POST['warning'])) {
            $warning_key = sanitize_text_field(wp_unslash($_POST['warning'])) . '_dismissed';
        }
    }

    // Only permanently dismiss warnings if needed
    update_user_meta($user_id, $warning_key, true);

    wp_send_json_success(['message' => __('Warning dismissed', 'wordsentinel')]);
    exit; // Ensure script execution stops
}

function NEXSOL_WS_reset_warning() {
    $user_id = get_current_user_id();
    if (!$user_id) {
        wp_send_json_error(['message' => __('User not logged in', 'wordsentinel')]);
    }

    // Delete the dismissed warning meta
    delete_user_meta($user_id, 'ws_update_warning_dismissed');
    wp_send_json_success(['message' => __('Warning reset successfully', 'wordsentinel')]);
}

function NEXSOL_WS_fetch_grades() {
    require_once NEXSOL_WS_PLUGIN_DIR . 'includes/api-helper.php';

    $apiHelper = new NEXSOL_WS_API_Helper();
    $siteUrl = wp_parse_url(get_site_url(), PHP_URL_HOST);

    $results = [
        'Mozilla Observatory' => $apiHelper->fetch_mozilla_observatory_grade($siteUrl),
        'Qualys SSL Labs'     => $apiHelper->fetch_ssl_labs_grade($siteUrl),
    ];

    return rest_ensure_response($results);
}

function NEXSOL_WS_sanitize_headers_security($input) {
    if (!is_array($input)) {
        return [];
    }

    $sanitized = [];
    foreach ($input as $key => $value) {
        $sanitized[$key] = is_string($value) ? sanitize_text_field($value) : $value;
    }

    return $sanitized;
}

class NEXSOL_WS_Wordsentinel_Plugin {
    public function __construct() {
        remove_action('wp_head', 'wp_generator'); // Remove WordPress generator meta tag

        // Load dependencies
        require_once NEXSOL_WS_PLUGIN_DIR . 'includes/api-helper.php';

        // Hooks
        add_action('send_headers', [$this, 'set_security_headers']);
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
    }

    public function add_settings_page() {
        add_menu_page(
            __('WordSentinel', 'wordsentinel'),
            __('WordSentinel', 'wordsentinel'),
            'manage_options',
            'NEXSOL_WS_wordsentinel',
            [$this, 'render_plugin_page'],
            'data:image/svg+xml;base64,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',
            90
        );
    }

    public function render_plugin_page() {
        ?>
        <div class="wrap wordsentinel-plugin">
            <!-- Header Section -->
            <div class="wordsentinel-header">
                <div class="title">
                    <div class="wordsentinel-logo"></div>
                    <span class="word"><?php echo esc_html__('Word', 'wordsentinel'); ?></span>
                    <span class="sentinel"><?php echo esc_html__('Sentinel', 'wordsentinel'); ?></span>
                </div>

                <?php
                $user_id = get_current_user_id();
                $warnings = [
                    'ws_general_warning' => __('It is advised to disable any other header functionalities to allow WordSentinel to work its magic.', 'wordsentinel'),
                    'ws_update_warning'  => __('Make sure to clear any cache after installing WordSentinel or changing settings.', 'wordsentinel')
                ];

                foreach ($warnings as $key => $message) {
                    $meta_key = $key . '_dismissed';
                    $dismissed = get_user_meta($user_id, $meta_key, true);

                    if (!$dismissed) : ?>
                        <div class="warning-notice" data-warning="<?php echo esc_attr($key); ?>">
                            <span class="dashicons dashicons-warning"></span>
                            <span class="message"><?php echo esc_html($message); ?></span>
                            <button class="button button-primary"><?php esc_html_e('OK', 'wordsentinel'); ?></button>
                        </div>
                    <?php endif;
                }
                ?>
            </div>
    
            <!-- Dashboard Section -->
            <div class="wordsentinel-dashboard">
                <div class="wordsentinel-tooltip-container">
                    <h2><?php echo esc_html__('Dashboard', 'wordsentinel'); ?></h2>
                    <!-- Tooltip -->
                    <div class="wordsentinel-tooltip">
                        <span class="wordsentinel-tooltip-icon">?</span>
                        <div class="wordsentinel-tooltip-content">
                            <table>
                                <thead>
                                    <tr>
                                        <th><?php echo esc_html__('Grade', 'wordsentinel'); ?></th>
                                        <th><?php echo esc_html__('Score', 'wordsentinel'); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr><td>A+</td><td>100+</td></tr>
                                    <tr><td>A</td><td>90</td></tr>
                                    <tr><td>A−</td><td>85</td></tr>
                                    <tr><td>B+</td><td>80</td></tr>
                                    <tr><td>B</td><td>70</td></tr>
                                    <tr><td>B−</td><td>65</td></tr>
                                    <tr><td>C+</td><td>60</td></tr>
                                    <tr><td>C</td><td>50</td></tr>
                                    <tr><td>C-</td><td>45</td></tr>
                                    <tr><td>D+</td><td>40</td></tr>
                                    <tr><td>D</td><td>30</td></tr>
                                    <tr><td>D-</td><td>25</td></tr>
                                    <tr><td>F</td><td>0</td></tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
    
                <h4><?php echo esc_html__('These are your current grades on two renowned and trusted services:', 'wordsentinel'); ?></h4>
                    
                <div class="wordsentinel-panels">
                    <div class="wordsentinel-panel wordsentinel-loading" id="panel-mozilla">
                        <h3>Mozilla Observatory</h3>
                        <div class="wordsentinel-loading">
                            <div class="wordsentinel-loading-spinner"></div>
                        </div>
                    </div>
                    <div class="wordsentinel-panel wordsentinel-loading" id="panel-ssl-labs">
                        <h3>Qualys SSL Labs</h3>
                        <div class="wordsentinel-loading">
                            <div class="wordsentinel-loading-spinner"></div>
                        </div>
                    </div>
                </div>

                <h3><?php echo esc_html__('Exciting News!', 'wordsentinel'); ?></h3>
                <span></span><?php echo esc_html__("We're working on a premium version of this plugin that will bring you even more powerful features to enhance your experience. With the premium version, you'll be able to access to:", 'wordsentinel'); ?></span>
                <p><span class="dashicons dashicons-yes"></span><?php echo esc_html__('Advanced Security Scanning - Get in-depth reports and automated security analysis.', 'wordsentinel'); ?></p>
                <p><span class="dashicons dashicons-yes"></span><?php echo esc_html__('Enhanced CSP Management - Intelligent resource discovery and seamless CSP configuration.', 'wordsentinel'); ?></p>
                <p><span class="dashicons dashicons-yes"></span><?php echo esc_html__('Automated Security Reports - Receive periodic security insights straight to your inbox.', 'wordsentinel'); ?></p>
                <p><span class="dashicons dashicons-yes"></span><?php echo esc_html__('Cache Management - Detect and flush caching systems for optimal security updates.', 'wordsentinel'); ?></p>
                <p class="submit">
                    <a href="https://nexsol-tech.ch/our-products/wordsentinel/" class="button button-primary" target="_blank">
                        <?php echo esc_html__( 'Learn more...', 'wordsentinel' ); ?>
                    </a>
                </p>
            </div>
    
            <!-- Settings Section --> 
            <div class="wordsentinel-settings">
                <form method="post" action="options.php" class="wordsentinel-settings-form">
                    <?php
                    wp_nonce_field('NEXSOL_WS_warning_action', 'NEXSOL_WS_warning_nonce'); 
                    settings_fields('NEXSOL_WS_headers_security_options'); // Group of options
                    do_settings_sections('NEXSOL_WS_wordsentinel');    // Sections of options
                    ?>
                    <div class="wordsentinel-settings-footer">
                        <button type="submit" class="button button-primary" disabled>
                            <?php echo esc_html__('Save Changes', 'wordsentinel'); ?>
                        </button>
                    </div>
                </form>
            </div>

            <!-- Footer Section -->
            <div class="wordsentinel-footer">
                <p>Powered by</p>
                <a href="https://nexsol-tech.ch/" target="_blank">
                    <div class="nexsol-logo"></div>
                </a>
            </div>
        </div>
        <?php
    }    

    public function register_settings() {
        register_setting(
            'NEXSOL_WS_headers_security_options',
            'NEXSOL_WS_headers_security',
            'NEXSOL_WS_sanitize_headers_security'
        );
    
        add_settings_section(
            'headers_security_section',
            __('Headers Settings', 'wordsentinel'),
            null,
            'NEXSOL_WS_wordsentinel'
        );
    
        $headers = [
            'strict_transport_security' => __('Strict-Transport-Security: Forces the use of HTTPS', 'wordsentinel'),
            'content_security_policy' => __('Content-Security-Policy: Controls allowed content sources', 'wordsentinel'),
            'x_content_type_options' => __('X-Content-Type-Options: Prevents MIME sniffing', 'wordsentinel'),
            'x_frame_options' => __('X-Frame-Options: Prevents clickjacking attacks', 'wordsentinel'),
            'referrer_policy' => __('Referrer-Policy: Controls referrer information', 'wordsentinel'),
            'permissions_policy' => __('Permissions-Policy: Limits features (camera, geolocation, etc.)', 'wordsentinel'),
            'coop' => __('Cross-Origin-Opener-Policy: Protects against cross-origin attacks', 'wordsentinel'),
            'coep' => __('Cross-Origin-Embedder-Policy: Secures embedded content', 'wordsentinel'),
            'xss' => __('X-XSS-Protection: Protection against XSS attacks', 'wordsentinel'),
            'expect_ct' => __('Expect-CT: Detect non-compliant SSL certificates', 'wordsentinel'),
            'cache_control' => __('Cache-Control & Pragma: Control content caching', 'wordsentinel'),
        ];
    
        foreach ($headers as $key => $description) {
            add_settings_field(
                $key,
                $description,
                [$this, 'display_checkbox_option_header'],
                'NEXSOL_WS_wordsentinel',
                'headers_security_section',
                ['label_for' => $key]
            );
        }
    }
    
    public function display_checkbox_option_header($args) {
        $option = get_option('NEXSOL_WS_headers_security', []);
        $key = $args['label_for'];
        ?>
        <label class="wordsentinel-toggle">
            <input type="checkbox" name="NEXSOL_WS_headers_security[<?php echo esc_html($key); ?>]" id="<?php echo esc_html($key); ?>" value="1" <?php checked(isset($option[$key]) && $option[$key] == 1); ?>>
            <span class="wordsentinel-slider"></span>
        </label>
        <?php
    }

    public function set_security_headers() {
        $options = get_option('NEXSOL_WS_headers_security', []);

        if (!empty($options['strict_transport_security'])) {
            header("Strict-Transport-Security: max-age=31536000; includeSubDomains; preload");
        }

        if (!empty($options['content_security_policy'])) {
            header("Content-Security-Policy: default-src 'self' localhost:*; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-hashes' 'unsafe-inline'; object-src 'none'; img-src 'self' blob: data:; font-src 'self' data:; worker-src 'self' blob:;");
        }

        if (!empty($options['x_content_type_options'])) {
            header("X-Content-Type-Options: nosniff");
        }

        // Add Set-Cookie header
        setcookie(
            'wordsentinel_cookie',
            'wordsentinel_ck_value',
            [
                'Path' => '/',
                'Secure' => is_ssl(),
                'HttpOnly' => true,
                'SameSite' => is_ssl() ? 'None' : 'Lax',
            ]
        );

        if (!empty($options['x_frame_options'])) {
            header("X-Frame-Options: SAMEORIGIN");
        }
        if (!empty($options['referrer_policy'])) {
            header("Referrer-Policy: strict-origin-when-cross-origin");
        }
        if (!empty($options['permissions_policy'])) {
            header("Permissions-Policy: geolocation=(self), microphone=()");
        }
        if (!empty($options['coop'])) {
            header("Cross-Origin-Opener-Policy: same-origin");
        }
        if (!empty($options['coep'])) {
            header("Cross-Origin-Embedder-Policy: unsafe-none");
        }
        if (!empty($options['xss'])) {
            header("X-XSS-Protection: 1; mode=block");
        }
        if (!empty($options['expect_ct'])) {
            header("Expect-CT: max-age=86400, enforce");
        }
        if (!empty($options['cache_control'])) {
            header("Cache-Control: no-cache, no-store, must-revalidate");
            header("Pragma: no-cache");
            header("Expires: 0");
        }
    }
}

new NEXSOL_WS_Wordsentinel_Plugin();
